<?php
/**
 * app/Views/matrices2/grid.php
 *
 * Vista tipo “Excel” para matrices:
 * - Carga total (hasta ~50k filas) por chunks adaptativos
 * - Columnas fijas base (Código, Descripción, Familia, Categorización)
 * - Columnas dinámicas desde la matriz (sin duplicar las base)
 * - Edición sólo de columnas marcadas como editables (no fórmula)
 * - Recalcula la fila al presionar Enter/terminar edición
 */

$matId   = (int)($mat['mat_id'] ?? 0);
$matTipo = esc($mat['mat_tipo'] ?? '');
$matNom  = esc($mat['mat_nombre'] ?? '');

/** Campos base fijos que ya pintamos en la grilla */
$BASE_PROPS = ['pro_codigo','pro_descripcion','pro_familia','pro_categorizacion'];

/** Construye dynCols desde $campos y elimina duplicados/colisiones con base */
$tmp = array_values(array_filter(array_map(function($c){
  if ((int)($c['mcr_visible'] ?? 1) !== 1) return null;
  return [
    'name'      => $c['cam_nombre'] ?? '',
    'title'     => ($c['mcr_titulo'] ?: ($c['cam_titulo'] ?: ($c['cam_nombre'] ?? ''))),
    'tipo'      => $c['cam_tipo'] ?? 'text',
    'origen'    => $c['cam_origen'] ?? 'lectura',
    'hasSource' => !empty($c['cam_source']),
    'editable'  => (bool)($c['editable'] ?? false),
  ];
}, $campos ?? [])));

$seen  = [];
$dynCols = [];
foreach ($tmp as $col) {
  $name = (string)($col['name'] ?? '');
  if ($name === '' || in_array($name, $BASE_PROPS, true)) continue; // quita duplicados con base
  if (isset($seen[$name])) continue;                                 // quita duplicados repetidos
  $seen[$name] = true;
  $dynCols[] = $col;
}
?>
<div class="container-fluid" style="margin-top:80px; --xl-blue:#1976d2; --xl-blue-50:#e8f2ff; --xl-blue-100:#d9eaff; --xl-border:#c7dbf7; --xl-grid:#e2ecfb;">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <div>
      <h2 class="mb-0" style="font-weight:700;color:var(--xl-blue);display:flex;align-items:center;gap:10px">
        <span style="display:inline-block;width:14px;height:24px;background:linear-gradient(180deg,var(--xl-blue),#42a5f5);border-radius:2px"></span>
        <?= $matNom ?>
        <small class="text-muted" style="font-weight:500"> (<?= $matTipo ?>)</small>
      </h2>
      <div class="text-muted" style="margin-top:4px">Vista tipo Excel — edita solo columnas marcadas como <b>editables</b>; las fórmulas se recalculan al guardar.</div>
    </div>
    <div class="d-flex gap-2">
      <button id="btn-export" class="btn btn-sm btn-outline-primary">Exportar CSV</button>
      <a class="btn btn-sm btn-outline-secondary" href="<?= site_url('matrices2'); ?>">Volver</a>
    </div>
  </div>

  <div class="card shadow-sm" style="border:1px solid var(--xl-grid); border-radius:12px; overflow:hidden">
    <div class="card-header" style="background:var(--xl-blue-50);border-bottom:1px solid var(--xl-grid);padding:10px 14px">
      <div class="d-flex align-items-center justify-content-between">
        <div class="d-flex align-items-center" style="gap:12px">
          <span class="badge" style="background:var(--xl-blue);color:white">Excel Grid</span>
          <span class="text-muted" style="font-size:.9rem">Carga total (hasta 50k) con scroll y recálculo por fila.</span>
        </div>
        <div id="mini-stats" class="text-muted" style="font-size:.9rem">0 filas cargadas</div>
      </div>
    </div>

    <div class="card-body p-0" style="position:relative">
      <!-- Overlay de carga -->
      <div id="grid-loader" style="position:absolute;inset:0;background:rgba(255,255,255,.88);display:flex;flex-direction:column;justify-content:center;align-items:center;z-index:5">
        <div class="spinner-border text-primary" role="status" style="width:2.5rem;height:2.5rem"></div>
        <div class="mt-3" style="width:min(520px,90%);">
          <div class="progress" style="height:8px">
            <div id="grid-progress" class="progress-bar" role="progressbar" style="width: 0%"></div>
          </div>
        </div>
        <div id="grid-progress-text" class="text-muted mt-2" style="font-size:.9rem">Preparando…</div>
      </div>

      <!-- Contenedor Handsontable -->
      <div id="matriz-grid" style="height: calc(100vh - 260px);"></div>
    </div>
  </div>

  <small class="text-muted d-block mt-2">
    <span style="display:inline-block;width:12px;height:12px;background:#fff;border:1px solid #ddd;margin-right:4px;vertical-align:middle"></span> Editable &nbsp;&nbsp;
    <span style="display:inline-block;width:12px;height:12px;background:#f9fbff;border:1px solid #e0e7ff;margin-right:4px;vertical-align:middle"></span> Fórmula/solo lectura
  </small>
</div>

<!-- ====== Assets ====== -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/handsontable@14.4.0/dist/handsontable.full.min.css" />
<script src="https://cdn.jsdelivr.net/npm/handsontable@14.4.0/dist/handsontable.full.min.js"></script>

<script src="https://cdn.jsdelivr.net/npm/numbro@2.3.6/dist/numbro.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/numbro@2.3.6/dist/languages.min.js"></script>
<script>
  (function(){
    var nb = window.numbro && (window.numbro.default || window.numbro);
    if (nb) { try { nb.setLanguage('es-ES'); } catch(e) {} }
  })();
</script>

<style>
  /* Estética Excel */
  #matriz-grid .handsontable .ht_clone_top th,
  #matriz-grid .handsontable .ht_clone_left th,
  #matriz-grid .handsontable th {
    background: linear-gradient(180deg, var(--xl-blue-50), var(--xl-blue-100));
    color:#0d47a1;
    font-weight:600;
    border-color:var(--xl-border) !important;
  }
  #matriz-grid .handsontable table,
  #matriz-grid .handsontable td,
  #matriz-grid .handsontable th {
    border-color:var(--xl-grid) !important;
  }
  #matriz-grid .handsontable td {
    padding:2px 6px;
    line-height:1.2;
  }
  .cell-manual { background:#fff !important; }
  .cell-manual:focus-within { outline: 2px solid rgba(25,118,210,.25); outline-offset:-2px; }
  .cell-formula { background:#f9fbff !important; color:#222; }
  .htRight { text-align:right !important; }
  .xl-col-badge {
    font-size:.72rem; font-weight:600; color:#0d47a1; background:#e9f2ff; border:1px solid #cfe1ff;
    padding:0 .35rem; border-radius:.35rem; margin-left:.35rem;
  }
</style>

<script>
(function () {
  // --- ENDPOINTS & CSRF ---
  const URL_DATA = '<?= site_url("matrices2/{$mat['mat_id']}/data") ?>';
  const URL_SAVE = '<?= site_url("matrices2/{$mat['mat_id']}/save-manual") ?>';
  const URL_EVAL = '<?= site_url("matrices2/{$mat['mat_id']}/eval") ?>';
  const CSRF_KEY = '<?= csrf_token() ?>';
  const CSRF_VAL = '<?= csrf_hash() ?>';

  // --- COLUMNAS DINÁMICAS (PHP -> JS) ---
  let dynCols = <?= json_encode($dynCols ?? [], JSON_UNESCAPED_UNICODE) ?>;

  // Limpieza defensiva de duplicados (por si llega algo sucio)
  (function(){
    const baseProps = new Set(['pro_codigo','pro_descripcion','pro_familia','pro_categorizacion']);
    const unique = [];
    const seen = new Set();
    for (const c of (dynCols || [])) {
      if (!c || !c.name) continue;
      const nm = String(c.name);
      if (baseProps.has(nm)) continue;
      if (seen.has(nm)) continue;
      seen.add(nm);
      unique.push(c);
    }
    dynCols = unique;
  })();

  // Campos editables (no-fórmula)
  const EDITABLE_FIELDS = new Set(
    (dynCols || [])
      .filter(c => !!c.editable && String(c.origen).toLowerCase() !== 'formula')
      .map(c => String(c.name))
  );

  // UI
  const $loader   = document.getElementById('grid-loader');
  const $progBar  = document.getElementById('grid-progress');
  const $progTxt  = document.getElementById('grid-progress-text');
  const $miniStat = document.getElementById('mini-stats');
  const $export   = document.getElementById('btn-export');
  const $mount    = document.getElementById('matriz-grid');

  // Formateadores
  const nf4 = new Intl.NumberFormat('es-EC', { minimumFractionDigits: 4, maximumFractionDigits: 4 });
  function fmtNumber4(v){ if(v==null||v==='') return ''; const n=Number(v); return isFinite(n)?nf4.format(n):''; }
  function fmtPercent4(v){ if(v==null||v==='') return ''; const n=Number(v); return isFinite(n)?nf4.format(n*100)+'%':''; }

  // Renderers
  function rightNumberRenderer(instance, td, row, col, prop, value) {
    Handsontable.renderers.TextRenderer.apply(this, arguments);
    td.classList.add('htRight'); td.textContent = fmtNumber4(value);
  }
  function percentRenderer(instance, td, row, col, prop, value) {
    Handsontable.renderers.TextRenderer.apply(this, arguments);
    td.classList.add('htRight'); td.textContent = fmtPercent4(value);
  }

  // Columnas base fijas
  const BASE_COLS = [
    { prop: 'pro_codigo',         title: 'Código',         width: 140 },
    { prop: 'pro_descripcion',    title: 'Descripción',    width: 320 },
    { prop: 'pro_familia',        title: 'Familia',        width: 160 },
    { prop: 'pro_categorizacion', title: 'Categorización', width: 150 },
  ];

  // Construcción columnas
  function buildColumns() {
    const cols = [];
    const headers = [];

    // pro_id oculto
    cols.push({ data: 'pro_id', readOnly: true, width: 1 });
    headers.push('');

    BASE_COLS.forEach(c => { cols.push({ data:c.prop, readOnly:true, width:c.width }); headers.push(c.title); });

    (dynCols || []).forEach(c => {
      const tipo = String(c.tipo || '').toLowerCase();
      const isEditable = EDITABLE_FIELDS.has(String(c.name));
      const isFormula  = String(c.origen || '').toLowerCase() === 'formula';

      let renderer = 'text', className = '';
      if (tipo === 'percent') { renderer = percentRenderer; className = 'htRight'; }
      else if (/(number|money)/i.test(tipo)) { renderer = rightNumberRenderer; className = 'htRight'; }

      cols.push({
        data: c.name,
        readOnly: isFormula ? true : !isEditable,
        editor: (isFormula ? false : (isEditable ? 'text' : false)),
        renderer, className: (isFormula ? 'cell-formula ' : 'cell-manual ') + className,
        width: 170,
      });

      let h = c.title || c.name;
      if (isFormula) h += ' <span class="xl-col-badge">fx</span>';
      else if (isEditable) h += ' <span class="xl-col-badge">ed</span>';
      headers.push(h);
    });

    return { columns: cols, headers };
  }

  // Estado
  let hot = null;
  let colIndexByProp = Object.create(null);

  function initGrid(initialRows) {
    const cfg = buildColumns();

    colIndexByProp = {};
    cfg.columns.forEach((c, i) => { if (c.data) colIndexByProp[c.data] = i; });

    hot = new Handsontable($mount, {
      data: initialRows,
      columns: cfg.columns,
      colHeaders: cfg.headers,
      rowHeaders: true,
      licenseKey: 'non-commercial-and-evaluation',
      height: 'calc(100vh - 260px)',
      stretchH: 'all',
      manualColumnResize: true,
      manualRowResize: false,
      columnSorting: true,
      autoWrapRow: false,
      autoWrapCol: false,
      wordWrap: false,
      className: 'htCenter',
      afterChange: onCellChange
    });

    // compact
    const style = document.createElement('style');
    style.textContent = `
      .handsontable .ht_master .wtHolder { scrollbar-gutter: stable; }
      .handsontable td, .handsontable th { padding: 4px 6px; line-height: 1.2; }
      .handsontable col.header, .handsontable th { white-space: nowrap; }
      .handsontable .colHeader { font-size: .88rem; }
    `;
    document.head.appendChild(style);
  }

  // Normaliza valor
  function normalizeForSave(prop, value) {
    const colDef = (dynCols || []).find(c => c.name === prop);
    if (!colDef) return value;
    const tipo = String(colDef.tipo || '').toLowerCase();

    if (tipo === 'percent') {
      if (value == null || value === '') return null;
      const raw = String(value).replace('%','').replace(/\s+/g,'').replace(',','.');
      const num = Number(raw); return isFinite(num) ? (num/100.0) : null;
    }
    if (tipo === 'number' || tipo === 'money') {
      if (value == null || value === '') return null;
      const raw = String(value).replace(',','.'); const num = Number(raw);
      return isFinite(num) ? num : null;
    }
    return value;
  }

  // Guardar + Recalc
  async function onCellChange(changes, source) {
    if (!changes || source === 'loadData' || source === 'recalc') return;
    const toSave = [];
    for (const [row, prop, oldVal, newVal] of changes) {
      if (prop && EDITABLE_FIELDS.has(String(prop)) && oldVal !== newVal) {
        const rowData = hot.getSourceDataAtRow(row);
        if (!rowData || !rowData.pro_id) continue;
        const norm = normalizeForSave(prop, newVal);
        toSave.push({ pro_id: rowData.pro_id, name: prop, value: norm });
      }
    }
    if (!toSave.length) return;

    try {
      for (const item of toSave) {
        await saveManual(item.pro_id, item.name, item.value);
        const recalc = await evalRow(item.pro_id, { [item.name]: item.value });
        if (recalc && recalc.row) updateRowByProId(item.pro_id, recalc.row);
      }
    } catch (err) {
      console.error('save/eval error', err);
      alert('Ocurrió un error al guardar o recalcular:\n' + (err?.message || err));
    }
  }

  async function saveManual(proId, name, value) {
    const body = new URLSearchParams();
    body.append('pro_id', proId);
    body.append('name', name);
    body.append('value', value === null ? '' : String(value));
    body.append(CSRF_KEY, CSRF_VAL);
    const r = await fetch(URL_SAVE, { method: 'POST', body });
    if (!r.ok) throw new Error('save HTTP ' + r.status);
    const j = await r.json();
    if (!j.ok) throw new Error('save not ok');
    return true;
  }

  async function evalRow(proId, overridesObj) {
    const body = new URLSearchParams();
    body.append('pro_id', proId);
    body.append('overrides', JSON.stringify(overridesObj || {}));
    body.append(CSRF_KEY, CSRF_VAL);
    const r = await fetch(URL_EVAL, { method: 'POST', body });
    if (!r.ok) throw new Error('eval HTTP ' + r.status);
    return r.json();
  }

  function updateRowByProId(proId, newRow) {
    const data = hot.getSourceData();
    const idx  = data.findIndex(r => String(r.pro_id) === String(proId));
    if (idx < 0) return;
    Object.keys(newRow).forEach(prop => {
      if (prop in colIndexByProp) {
        const col = colIndexByProp[prop];
        hot.setDataAtCell(idx, col, newRow[prop], 'recalc');
      }
    });
  }

  function exportCSV() {
    if (!hot) return;
    const headers = hot.getColHeader().map(h => (typeof h === 'string') ? h.replace(/<[^>]+>/g,'') : h);
    const out = [headers.join(';')];
    const data = hot.getData();
    for (const row of data) out.push(row.map(v => (v == null ? '' : String(v).replace(/;/g, ','))).join(';'));
    const blob = new Blob([out.join('\n')], { type: 'text/csv;charset=utf-8' });
    const a = document.createElement('a'); a.href = URL.createObjectURL(blob); a.download = 'matriz_<?= $matId ?>.csv';
    document.body.appendChild(a); a.click(); a.remove();
  }
  if ($export) $export.addEventListener('click', exportCSV);

  // -------- CARGA TOTAL (chunks adaptativos) --------
  let CHUNK_SIZE = 5000;
  const MIN_CHUNK = 500;
  const MAX_RETRY = 3;

  async function fetchChunk(start, length, attempt = 1) {
    const body = new URLSearchParams();
    body.append('draw','1'); body.append('start', String(start)); body.append('length', String(length));
    body.append('search[value]',''); body.append(CSRF_KEY, CSRF_VAL);
    try {
      const r = await fetch(URL_DATA, { method:'POST', body });
      if (!r.ok) throw new Error('HTTP ' + r.status);
      return await r.json();
    } catch (err) {
      if (attempt < MAX_RETRY && length > MIN_CHUNK) {
        const nextLen = Math.max(MIN_CHUNK, Math.floor(length*0.66));
        await new Promise(res => setTimeout(res, 500*attempt));
        return fetchChunk(start, nextLen, attempt+1);
      }
      throw err;
    }
  }

  async function loadAll() {
    if ($loader) $loader.style.display = 'flex';
    if ($progTxt) $progTxt.textContent = 'Cargando…';
    if ($progBar) $progBar.style.width = '0%';

    let start = 0, total = null, loaded = 0, rows = [];

    while (true) {
      const resp = await fetchChunk(start, CHUNK_SIZE);
      const data = resp.data || [];
      if (total === null) total = resp.recordsFiltered || resp.recordsTotal || data.length;

      if (data.length) {
        rows = rows.concat(data);
        loaded += data.length;

        if (!hot) initGrid(rows);
        else hot.updateData(rows);

        const pct = Math.min(100, Math.round((loaded / Math.max(1, total)) * 100));
        if ($progBar) $progBar.style.width = pct + '%';
        if ($progTxt) $progTxt.textContent = `Cargando ${loaded.toLocaleString()} / ${total.toLocaleString()}…`;
        if ($miniStat) $miniStat.textContent = `${loaded.toLocaleString()} filas cargadas`;
      }

      if (data.length === 0) break;
      start += data.length;
      if (loaded >= (total ?? 0)) break;
    }

    if ($loader) $loader.style.display = 'none';
  }

  loadAll();
})();
</script>
