<?php
namespace App\Commands;

use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;
use App\Services\PrecioApiService;
use App\Models\ProductoModel;
use App\Models\ProductoStockModel;
use App\Models\SyncLogProductoModel;

class SyncProductos extends BaseCommand
{
    protected $group       = 'precios';
    protected $name        = 'precios:sync:productos';
    protected $description = 'Sincroniza productos desde API Megaprofer (search-products)';


    public function run(array $params)
    {
        $pageSize   = 500;       // acordado
        $maxRetries = 2;
        $svc        = new PrecioApiService();
        $mProd      = new ProductoModel();
        $mStock     = new ProductoStockModel();
        $mLog       = new SyncLogProductoModel();

        $startedAt  = date('Y-m-d H:i:s');
        $logId = $mLog->insert([
            'started_at' => $startedAt,
            'page_size'  => $pageSize,
            'status'     => 'RUNNING',
            'message'    => '',
        ], true);

        $totPages = 0; $totElems = 0; $ins = 0; $upd = 0; $inact = 0; $errs = 0;

        try {
            // Página 0 para obtener total
            $first = $this->fetchWithRetry($svc, '', 0, $pageSize, $maxRetries, $errs);
            $totPages = (int)($first['totalPages'] ?? 0);
            $totElems = (int)($first['totalElements'] ?? 0);

            // Marca de corrida: usamos $startedAt
            $runMark = $startedAt;

            // Proceso página 0..totPages-1
            for ($p = 0; $p < $totPages; $p++) {
                if ($p > 0) {
                    $page = $this->fetchWithRetry($svc, '', $p, $pageSize, $maxRetries, $errs);
                } else {
                    $page = $first;
                }

                $data = $page['data'] ?? [];
                if (empty($data)) continue;

                // mapear y chunk para upsert
                $rows = [];
                $stocks = [];
                foreach ($data as $it) {
                    $refId   = (string)($it['referenceId'] ?? '');
                    $code    = (string)($it['referenceCode'] ?? '');
                    $barcode = (string)($it['barcode'] ?? '');
                    $name    = (string)($it['name'] ?? '');
                    $desc    = (string)($it['description'] ?? '');

                    $tax     = $it['taxPresenter'] ?? [];
                    $unit    = $it['measureUnitPresenter'] ?? [];
                    $subline = $it['productSublinePresenter'] ?? [];
                    $familyP = $subline['productFamilyPresenter'] ?? [];
                    $mark    = $it['markPresenter'] ?? [];
                    $stocksA = $it['stocks'] ?? [];

                    $row = [
                        'pro_reference_id'        => $refId,
                        'pro_codigo'              => $code !== '' ? $code : null,
                        'pro_barcode'             => $barcode !== '' ? $barcode : null,
                        'pro_descripcion'         => $name,
                        'pro_detalle'             => $desc,
                        'pro_familia'             => (string)($familyP['name'] ?? ($it['family'] ?? '')),
                        'pro_sublinea'            => (string)($subline['name'] ?? ''),
                        'pro_marca'               => (string)($mark['name'] ?? ''),
                        'pro_proveedor_habitual'  => (string)($it['provider'] ?? ''),
                        'pro_categorizacion'      => (string)($it['categorization'] ?? ''),
                        'pro_unidad_nombre'       => (string)($unit['name'] ?? ''),
                        'pro_unidad_simbolo'      => (string)($unit['symbol'] ?? ''),
                        'pro_iva_nombre'          => (string)($tax['name'] ?? ''),

                        'pro_peso'                => $it['weight'] ?? null,
                        'pro_iva_valor'           => $tax['value'] ?? null,
                        'pro_cost'                => $it['cost'] ?? null,
                        'pro_price_cost'          => $it['priceCost'] ?? null,
                        'pro_payed_price'         => $it['payedPrice'] ?? null,
                        'pro_stock'               => $it['stock'] ?? null,
                        'pro_authorized_discount' => $it['authorizedDiscount'] ?? null,

                        'pro_activo'              => true,
                        'pro_es_pesado'           => $it['isHeavy'] ?? null,
                        'pro_es_especial'         => $it['isEspecial'] ?? null,
                        'pro_need_validate_stock' => $it['needValidateStock'] ?? null,
                        'pro_is_supply'           => $it['isSupply'] ?? null,

                        'pro_location'            => (string)($it['location'] ?? ''),
                        'pro_updated_api'         => !empty($it['updatedDate']) ? $it['updatedDate'] : null,
                        'pro_seen_at'             => $runMark,
                        'pro_inactive_at'         => null,
                        'pro_json'                => json_encode($it, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
                    ];

                    // Normaliza textos: sin NULLs
                    foreach (['pro_descripcion','pro_detalle','pro_familia','pro_sublinea','pro_marca',
                              'pro_proveedor_habitual','pro_categorizacion','pro_unidad_nombre',
                              'pro_unidad_simbolo','pro_iva_nombre','pro_location'] as $k) {
                        if ($row[$k] === null) $row[$k] = '';
                    }

                    // Hash de comparación (solo campos relevantes)
                    $hashBase = [
                        $row['pro_codigo'],$row['pro_barcode'],$row['pro_descripcion'],$row['pro_detalle'],
                        $row['pro_familia'],$row['pro_sublinea'],$row['pro_marca'],$row['pro_proveedor_habitual'],
                        $row['pro_categorizacion'],$row['pro_unidad_nombre'],$row['pro_unidad_simbolo'],$row['pro_iva_nombre'],
                        $row['pro_peso'],$row['pro_iva_valor'],$row['pro_cost'],$row['pro_price_cost'],$row['pro_payed_price'],
                        $row['pro_stock'],$row['pro_authorized_discount'],$row['pro_es_pesado'],$row['pro_es_especial'],
                        $row['pro_need_validate_stock'],$row['pro_is_supply'],$row['pro_location'],$row['pro_updated_api']
                    ];
                    $row['pro_hash'] = sha1(json_encode($hashBase));

                    $rows[] = $row;

                    // stocks por bodega
                    if (is_array($stocksA)) {
                        foreach ($stocksA as $st) {
                            $wName = (string)($st['warehouseName'] ?? '');
                            if ($wName === '') continue;
                            $stocks[] = [
                                'pro_reference_id' => $refId,
                                'warehouse_name'   => $wName,
                                'stock'            => $st['stock'] ?? 0
                            ];
                        }
                    }
                }

                // Transacción por página
                $db = \Config\Database::connect('postgres');
                $db->transStart();

                $res = $mProd->upsertBatchWithHash($rows);
                $ins += $res['inserted'] ?? 0;
                $upd += $res['updated'] ?? 0;

                // UPSERT stocks
                $mStock->upsertStocks($stocks);

                $db->transComplete();
                if (!$db->transStatus()) {
                    $errs++;
                }

                CLI::write("Página {$p}/{$totPages} — +{$res['inserted']} ins, +{$res['updated']} upd");
            }

            // Inactivar no vistos en esta corrida
            $inact = $mProd->inactivateNotSeenSince($runMark);

            $mLog->update($logId, [
                'finished_at'    => date('Y-m-d H:i:s'),
                'page_size'      => $pageSize,
                'total_pages'    => $totPages,
                'total_elements' => $totElems,
                'inserted'       => $ins,
                'updated'        => $upd,
                'inactivated'    => $inact,
                'errors'         => $errs,
                'status'         => $errs > 0 ? 'PARTIAL' : 'OK',
                'message'        => 'Sync completa'
            ]);

            CLI::write("OK: ins={$ins}, upd={$upd}, inact={$inact}, errors={$errs}");

        } catch (\Throwable $e) {
            $mLog->update($logId, [
                'finished_at'    => date('Y-m-d H:i:s'),
                'status'         => 'ERROR',
                'message'        => $e->getMessage(),
            ]);
            CLI::error("ERROR: ".$e->getMessage());
        }
    }

    private function fetchWithRetry(PrecioApiService $svc, string $search, int $page, int $size, int $maxRetries, int &$errCounter): array
    {
        $attempt = 0;
        while (true) {
            try {
                return $svc->searchProductsPaged($search, $page, $size);
            } catch (\Throwable $e) {
                $attempt++;
                $errCounter++;
                if ($attempt > $maxRetries) throw $e;
                sleep($attempt === 1 ? 2 : 5);
            }
        }
    }
}
