<?php
namespace App\Controllers;

use CodeIgniter\Controller;
use CodeIgniter\Database\BaseConnection;

class MatrizUI extends BaseController
{
    /** @var BaseConnection */
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect('postgres');
    }

    /* ------------------------- Helpers ------------------------- */

    /** Normaliza el campo "mac_depends" (array o string) y devuelve un array simple */
    private function normDepends($raw): array
    {
        if (is_array($raw)) {
            $arr = $raw;
        } elseif (is_string($raw) && trim($raw) !== '') {
            // por si llegara “a,b,c” desde algún cliente viejo
            $arr = array_map('trim', explode(',', $raw));
        } else {
            $arr = [];
        }
        // filtra vacíos y fuerza string
        $arr = array_values(array_filter(array_map(fn($v) => (string)$v, $arr), fn($v) => $v !== ''));
        return $arr;
    }

    /** Decodifica JSON seguro y devuelve array|mixed o null si falla */
    private function jsonDecode($val)
    {
        if ($val === null || $val === '') return null;
        if (is_array($val)) return $val;
        if (!is_string($val)) return null;
        $j = json_decode($val, true);
        return (json_last_error() === JSON_ERROR_NONE) ? $j : null;
    }

    /* ------------------------- UI ------------------------- */

    public function index()
    {
        $mats = $this->db->table('public.tbl_matriz')
            ->select('mat_id, mat_tipo, mat_nombre, mat_estado')
            ->orderBy('mat_id','asc')->get()->getResultArray();

        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('matrices/index', ['matrices' => $mats]);
        echo view('layouts/footer');
    }

    public function campos($matId)
    {
        $mat = $this->db->table('public.tbl_matriz')
            ->where('mat_id', (int)$matId)->get()->getRowArray();
        if (!$mat) return redirect()->to('/matrices');

        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('matrices/campos', ['mat' => $mat]);
        echo view('layouts/footer');
    }

    /* ------------------------- Campos: CRUD (AJAX) ------------------------- */

    public function camposList($matId)
    {
        $rows = $this->db->table('public.tbl_matriz_campo')
            ->where('mat_id', (int)$matId)
            ->orderBy('mac_orden IS NULL, mac_orden ASC, mac_id ASC', '', false)
            ->get()->getResultArray();

        // Decodificar json para el front (comodidad del <select multiple>)
        $rows = array_map(function($r){
            $r['mac_depends'] = $this->jsonDecode($r['mac_depends']) ?? [];
            // mac_source mantener como string JSON (si existe) para el textarea
            if (is_array($r['mac_source'])) {
                $r['mac_source'] = json_encode($r['mac_source'], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
            }
            return $r;
        }, $rows);

        return $this->response->setJSON(['ok'=>true,'data'=>$rows]);
    }

    public function campoSave($matId)
    {
        $p    = $this->request->getPost();
        $deps = $this->normDepends($p['mac_depends'] ?? []);

        $data = [
            'mat_id'           => (int)$matId,
            'mac_nombre'       => trim($p['mac_nombre'] ?? ''),
            'mac_titulo'       => trim($p['mac_titulo'] ?? ''),
            'mac_tipo'         => trim($p['mac_tipo']   ?? 'text'),
            'mac_formato'      => trim($p['mac_formato']?? null),
            'mac_color'        => trim($p['mac_color']  ?? null),
            'mac_orden'        => ($p['mac_orden'] === '' ? null : (int)$p['mac_orden']),
            'mac_visible'      => (int)($p['mac_visible'] ?? 1),

            'mac_origen'       => trim($p['mac_origen'] ?? 'lectura'),
            'mac_valor_default'=> ($p['mac_valor_default'] ?? null),
            'mac_formula'      => ($p['mac_formula'] ?? null),

            // ✅ JSON SIEMPRE
            'mac_depends'      => json_encode($deps, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ];

        // mac_source (JSON opcional)
        $src = trim($p['mac_source'] ?? '');
        if ($src !== '') {
            $json = json_decode($src, true);
            if (!is_array($json)) {
                return $this->response->setStatusCode(422)
                    ->setJSON(['ok'=>false,'msg'=>'mac_source debe ser JSON válido']);
            }
            $data['mac_source'] = json_encode($json, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        } else {
            $data['mac_source'] = null;
        }

        if ($data['mac_nombre']==='' || $data['mac_titulo']==='') {
            return $this->response->setStatusCode(422)
                ->setJSON(['ok'=>false,'msg'=>'Nombre y Título son obligatorios']);
        }

        try{
            $this->db->table('public.tbl_matriz_campo')->insert($data);
            return $this->response->setJSON(['ok'=>true,'id'=>$this->db->insertID()]);
        } catch(\Throwable $e){
            return $this->response->setStatusCode(500)
                ->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    public function campoUpdate($matId)
    {
        $p  = $this->request->getPost();
        $id = (int)($p['mac_id'] ?? 0);
        if(!$id) return $this->response->setStatusCode(400)
            ->setJSON(['ok'=>false,'msg'=>'mac_id requerido']);

        $deps = $this->normDepends($p['mac_depends'] ?? []);

        $data = [
            'mac_titulo'       => trim($p['mac_titulo'] ?? ''),
            'mac_tipo'         => trim($p['mac_tipo']   ?? 'text'),
            'mac_formato'      => trim($p['mac_formato']?? null),
            'mac_color'        => trim($p['mac_color']  ?? null),
            'mac_orden'        => ($p['mac_orden'] === '' ? null : (int)$p['mac_orden']),
            'mac_visible'      => (int)($p['mac_visible'] ?? 1),

            'mac_origen'       => trim($p['mac_origen'] ?? 'lectura'),
            'mac_valor_default'=> ($p['mac_valor_default'] ?? null),
            'mac_formula'      => ($p['mac_formula'] ?? null),

            // ✅ JSON SIEMPRE
            'mac_depends'      => json_encode($deps, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ];

        $src = trim($p['mac_source'] ?? '');
        if ($src !== '') {
            $json = json_decode($src, true);
            if (!is_array($json)) {
                return $this->response->setStatusCode(422)
                    ->setJSON(['ok'=>false,'msg'=>'mac_source debe ser JSON válido']);
            }
            $data['mac_source'] = json_encode($json, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        } else {
            $data['mac_source'] = null;
        }

        try{
            $this->db->table('public.tbl_matriz_campo')
                ->where(['mac_id'=>$id,'mat_id'=>(int)$matId])
                ->update($data);
            return $this->response->setJSON(['ok'=>true]);
        } catch(\Throwable $e){
            return $this->response->setStatusCode(500)
                ->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    public function campoDelete($matId)
    {
        $id = (int)($this->request->getPost('mac_id') ?? 0);
        if(!$id) return $this->response->setStatusCode(400)
            ->setJSON(['ok'=>false,'msg'=>'mac_id requerido']);
        try{
            $this->db->table('public.tbl_matriz_campo')
                ->where(['mac_id'=>$id,'mat_id'=>(int)$matId])
                ->delete();
            return $this->response->setJSON(['ok'=>true]);
        } catch(\Throwable $e){
            return $this->response->setStatusCode(500)
                ->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    /* ------------------------- Preferencias de usuario ------------------------- */

    public function saveUserView($matId)
    {
        $userId     = (int)service('auth')->user()->id; // ajusta a tu auth
        $columns    = $this->request->getPost('columns'); // JSON
        $pageLength = (int)($this->request->getPost('page_length') ?? 25);

        $row = $this->db->table('public.tbl_matriz_user_view')
            ->where(['user_id'=>$userId,'mat_id'=>(int)$matId])
            ->get()->getRowArray();

        $payload = [
            'columns'     => json_decode($columns,true) ?? [],
            'page_length' => $pageLength,
            'updated_at'  => date('c')
        ];

        if ($row){
            $this->db->table('public.tbl_matriz_user_view')
                ->where('muv_id',$row['muv_id'])->update($payload);
        } else {
            $payload += [
                'user_id'=>$userId,'mat_id'=>(int)$matId,'created_at'=>date('c')
            ];
            $this->db->table('public.tbl_matriz_user_view')->insert($payload);
        }
        return $this->response->setJSON(['ok'=>true]);
    }

    public function getUserView($matId)
    {
        $userId = (int)service('auth')->user()->id; // ajusta a tu auth
        $row = $this->db->table('public.tbl_matriz_user_view')
            ->where(['user_id'=>$userId,'mat_id'=>(int)$matId])
            ->get()->getRowArray();
        return $this->response->setJSON(['ok'=>true,'data'=>$row ?: null]);
    }

    /* ------------------------- Data de la matriz (Server-Side) ------------------------- */

    public function datos($matId)
    {
        $req    = $this->request;
        $draw   = (int)($req->getGet('draw') ?? 1);
        $start  = (int)($req->getGet('start') ?? 0);
        $len    = (int)($req->getGet('length') ?? 25);
        $search = trim($req->getGet('search')['value'] ?? '');

        $mat = $this->db->table('public.tbl_matriz')->where('mat_id',(int)$matId)->get()->getRowArray();
        if(!$mat) return $this->response->setJSON(['draw'=>$draw,'recordsTotal'=>0,'recordsFiltered'=>0,'data'=>[]]);

        $campos = $this->db->table('public.tbl_matriz_campo')
            ->where('mat_id',(int)$matId)
            ->orderBy('mac_orden IS NULL, mac_orden ASC, mac_id ASC','',false)
            ->get()->getResultArray();

        $b = $this->db->table('public.tbl_producto p')
            ->select('p.pro_id, p.pro_codigo, p.pro_descripcion, p.pro_categorizacion, p.pro_marca, p.pro_familia,
                      d.mad_id, d.mad_campos')
            ->join('public.tbl_matriz_dato d', 'd.pro_id = p.pro_id AND d.mat_id = '.(int)$matId, 'left')
            ->where("COALESCE(p.pro_activo,'true') = 'true'");

        if ($mat['mat_tipo']==='PORTAFOLIO') {
            $b->whereIn('p.pro_categorizacion', ['PORTAFOLIO','PORT ESPEC','PORT-PUNTU']);
        } elseif ($mat['mat_tipo']==='NO PORTAFOLIO') {
            $b->whereIn('p.pro_categorizacion', ['FUERA-PORT','ADMINISTRA','MKT']);
        } elseif ($mat['mat_tipo']==='CONVERSIONES') {
            $b->where("1=1");
        } elseif ($mat['mat_tipo']==='PUNTUALES') {
            $b->where("1=1");
        }

        if ($search !== '') {
            $b->groupStart()
                ->like('p.pro_codigo', $search)
                ->orLike('p.pro_descripcion', $search)
                ->orLike('p.pro_marca', $search)
                ->orLike('p.pro_familia', $search)
              ->groupEnd();
        }

        $total = $b->countAllResults(false);
        $b->orderBy('p.pro_codigo','asc')->limit($len, $start);
        $rows = $b->get()->getResultArray();

        $data = [];
        foreach ($rows as $r) {
            $line = [
                'pro_codigo'        => $r['pro_codigo'],
                'pro_descripcion'   => $r['pro_descripcion'],
                'pro_categorizacion'=> $r['pro_categorizacion'],
            ];
            $mad = is_array($r['mad_campos']) ? $r['mad_campos']
                 : ($this->jsonDecode($r['mad_campos']) ?? []);

            foreach ($campos as $c) {
                $name = $c['mac_nombre'];
                $line[$name] = $mad[$name] ?? null;
            }
            $data[] = $line;
        }

        return $this->response->setJSON([
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data,
        ]);
    }

    /* ------------------------- Probar valor de campo ------------------------- */

    // POST /matrices/{matId}/campos/test-value  (mac_id, pro_id, pdv_id?)
    public function campoTestValue($matId)
    {
        $macId = (int)($this->request->getPost('mac_id') ?? 0);
        $proId = (int)($this->request->getPost('pro_id') ?? 0);
        $pdvId = (int)($this->request->getPost('pdv_id') ?? 0);

        if (!$macId || !$proId) {
            return $this->response->setStatusCode(422)
                ->setJSON(['ok'=>false,'msg'=>'mac_id y pro_id son requeridos']);
        }

        $campo = $this->db->table('public.tbl_matriz_campo')
            ->where(['mac_id'=>$macId,'mat_id'=>(int)$matId])->get()->getRowArray();
        if (!$campo) return $this->response->setStatusCode(404)
            ->setJSON(['ok'=>false,'msg'=>'Campo no encontrado']);

        $prod = $this->db->table('public.tbl_producto')
            ->where('pro_id',$proId)->get()->getRowArray();
        if (!$prod) return $this->response->setStatusCode(404)
            ->setJSON(['ok'=>false,'msg'=>'Producto no encontrado']);

        $ctx = [
            'pp_actual'               => $prod['pro_cost'] ?? null,
            'precio_aut_mis_pedidos'  => $prod['pro_price_cost'] ?? null,
        ];

        // Buscar overrides/param en mad_campos (con/sin PDV)
        $b = $this->db->table('public.tbl_matriz_dato')
            ->where('mat_id',(int)$matId)
            ->where('pro_id',$proId);

        if ($pdvId > 0) {
            $b->where('pdv_id',$pdvId);
        } else {
            // pdv_id IS NULL
            $b->where('pdv_id IS NULL', null, false);
        }
        $mad = $b->get()->getRowArray();

        $madCampos = [];
        if ($mad && !empty($mad['mad_campos'])) {
            $madCampos = is_array($mad['mad_campos'])
                ? $mad['mad_campos']
                : ($this->jsonDecode($mad['mad_campos']) ?? []);
        }

        try {
            $val = $this->resolveFieldValue((int)$matId, $campo, $prod, $pdvId, $madCampos, $ctx);
            return $this->response->setJSON(['ok'=>true,'value'=>$val]);
        } catch (\Throwable $e) {
            return $this->response->setStatusCode(500)
                ->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    /* ------------------------- Resolver valores ------------------------- */

    private function resolveFieldValue(
        int $matId, array $campo, array $prod, int $pdvId = 0, array $madCampos = [], array &$ctx = []
    ) {
        $nombre = $campo['mac_nombre'];

        if (array_key_exists($nombre, $ctx)) return $ctx[$nombre];

        $origen = $campo['mac_origen'] ?? 'lectura';

        if ($origen === 'param') {
            $val = $madCampos[$nombre] ?? $campo['mac_valor_default'] ?? null;
            $ctx[$nombre] = $this->castByTipo($val, $campo['mac_tipo'] ?? 'text');
            return $ctx[$nombre];
        }

        if ($origen === 'lectura') {
            $src = $campo['mac_source'] ? ($this->jsonDecode($campo['mac_source']) ?: []) : [];
            $val = $this->readFromSource($src, $prod, $pdvId);
            if ($val === null && isset($src['fallback']) && $src['fallback']==='tbl_producto.pro_cost') {
                $val = $prod['pro_cost'] ?? null;
            }
            $ctx[$nombre] = $this->castByTipo($val, $campo['mac_tipo'] ?? 'text');
            return $ctx[$nombre];
        }

        if ($origen === 'formula') {
            // Dependencias (puede venir string json)
            $deps = $this->jsonDecode($campo['mac_depends']) ?? ($campo['mac_depends'] ?? []);
            if (!is_array($deps)) $deps = [];

            $depVals = [];
            foreach ($deps as $depName) {
                $depCampo = $this->db->table('public.tbl_matriz_campo')
                    ->where(['mat_id'=>$matId,'mac_nombre'=>$depName])->get()->getRowArray();
                if (!$depCampo) { $depVals[$depName] = null; continue; }
                $depVals[$depName] = $this->resolveFieldValue($matId, $depCampo, $prod, $pdvId, $madCampos, $ctx);
            }

            $expr = trim($campo['mac_formula'] ?? '');
            if ($expr === '') { $ctx[$nombre] = null; return null; }

            $val = $this->evaluateFormulaSql($expr, $depVals);
            $ctx[$nombre] = $this->castByTipo($val, $campo['mac_tipo'] ?? 'text');
            return $ctx[$nombre];
        }

        $ctx[$nombre] = null;
        return null;
    }

    private function readFromSource(array $src, array $prod, int $pdvId = 0)
    {
        $table = $src['table']  ?? null;
        $col   = $src['column'] ?? null;
        if (!$table || !$col) return null;

        $allow = [
            'tbl_producto'       => ['pro_cost','pro_price_cost','pro_descripcion','pro_unidad_nombre','pro_categorizacion','pro_familia','pro_gancho','pro_proveedor_habitual','pro_barcode'],
            'tbl_producto_stock' => ['stock','updated_at'],
        ];
        if (!isset($allow[$table]) || !in_array($col, $allow[$table], true)) {
            return null;
        }

        if ($table === 'tbl_producto') {
            return $prod[$col] ?? null;
        }

        if ($table === 'tbl_producto_stock') {
            $row = $this->db->table('public.tbl_producto_stock')
                ->select("$col AS v")
                ->where('pro_reference_id', $prod['pro_reference_id'] ?? null)
                ->orderBy('updated_at','desc')
                ->get()->getRowArray();
            return $row['v'] ?? null;
        }

        return null;
    }

    private function evaluateFormulaSql(string $expr, array $vars)
{
    // Reemplazamos variables por '?' y vamos llenando $binds en orden.
    $binds = [];

    $safeExpr = preg_replace_callback(
        '/\b[a-zA-Z_][a-zA-Z0-9_]*\b/',
        function($m) use ($vars, &$binds) {
            $name = $m[0];

            // ¿Es una variable disponible?
            if (array_key_exists($name, $vars)) {
                $binds[] = $vars[$name];
                return '?'; // marcador posicional
            }

            // Permitir funciones whitelisted
            $upper = strtoupper($name);
            $allowedFuncs = ['COALESCE','NULLIF','ABS'];
            if (in_array($upper, $allowedFuncs, true)) return $upper;

            // Dejar números/identificadores que no son variables (será error si no corresponde)
            return $name;
        },
        $expr
    );

    // Validación básica de caracteres
    if (!preg_match('/^[0-9\ \t\+\-\*\/\.\,\(\)A-Za-z_?]+$/', $safeExpr)) {
        throw new \RuntimeException('Fórmula contiene caracteres no permitidos');
    }

    // Ejecutar con binds posicionales
    $query = $this->db->query('SELECT ('.$safeExpr.') AS v', $binds);
    $row = $query->getRowArray();
    return $row['v'] ?? null;
}


    private function castByTipo($val, string $tipo)
    {
        if ($val === null) return null;
        switch ($tipo) {
            case 'number':
            case 'money':
            case 'percent':
                return is_numeric($val) ? (float)$val : null;
            case 'bool':
                return in_array($val, [1,'1',true,'true','t','on'], true);
            case 'date':
                return (string)$val;
            default:
                return (string)$val;
        }
    }
}
