<?php
/**
 * @var array $mat    // mat_id, mat_tipo, mat_nombre
 * @var array $campos // filas de tbl_matriz_campo_ref join tbl_campo_global
 */
$matId   = (int)($mat['mat_id'] ?? 0);
$matTipo = esc($mat['mat_tipo'] ?? '');
$matNom  = esc($mat['mat_nombre'] ?? '');

// Prepara columnas dinámicas (respeta orden y títulos visibles)
$dynCols = array_values(array_map(function($c){
    $name   = $c['cam_nombre'] ?? '';
    $title  = $c['mcr_titulo'] ?: ($c['cam_titulo'] ?: $name);
    $origen = $c['cam_origen'] ?? 'lectura';
    $hasSrc = !empty($c['cam_source']); // si tiene source no es manual
    return [
        'name'      => $name,
        'title'     => $title,
        'origen'    => $origen,
        'isManual'  => (strtolower($origen)==='lectura' && !$hasSrc), // editable
    ];
}, $campos ?? []));
?>
<div class="container-fluid" style="margin-top:80px">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h2 class="mb-0">
      Matriz: <?= $matNom ?> <small class="text-muted">(<?= $matTipo ?>)</small>
    </h2>
    <div class="d-flex gap-2">
      <a class="btn btn-sm btn-outline-secondary" href="<?= site_url('matrices2'); ?>">Volver</a>
      <button id="btn-export" class="btn btn-sm btn-outline-success">Exportar CSV</button>
    </div>
  </div>

  <div class="card">
    <div class="card-body">
      <!-- Contenedor de la grilla tipo Excel -->
      <div id="matriz-grid"></div>

      <small class="text-muted d-block mt-2">
        Edite directamente en las celdas de <b>PP Calculado</b> u <b>Observación PP Calculado</b>. Los demás campos se recalculan automáticamente.
      </small>
    </div>
  </div>
</div>

<!-- ========== ASSETS TABULATOR (ligeros y sin conflicto con DataTables) ========== -->
<link rel="stylesheet" href="https://unpkg.com/tabulator-tables@5.5.2/dist/css/tabulator.min.css">
<script src="https://unpkg.com/tabulator-tables@5.5.2/dist/js/tabulator.min.js"></script>

<script>
(function(){
  // Endpoints
  const urlData = '<?= site_url("matrices2/{$matId}/data") ?>';
  const urlSave = '<?= site_url("matrices2/{$matId}/save-manual") ?>';
  const urlEval = '<?= site_url("matrices2/{$matId}/eval") ?>'; // si no lo tienes, el recálculo se omitirá

  // CSRF (CI4)
  const csrfKey = '<?= csrf_token() ?>';
  const csrfVal = '<?= csrf_hash() ?>';

  // Columnas dinámicas desde servidor
  const dynCols = <?= json_encode($dynCols, JSON_UNESCAPED_UNICODE) ?>;

  // Nombres “manuales” conocidos por negocio
  const manualNames = new Set(['pp_calculado','obs_pp_calculado']);

  // ======== Definición de columnas ========
  const baseCols = [
    { title:"pro_id",          field:"pro_id", visible:false },
    { title:"Código",          field:"pro_codigo",         headerFilter:"input", width:140 },
    { title:"Descripción",     field:"pro_descripcion",    headerFilter:"input", width:320 },
    { title:"Familia",         field:"pro_familia",        headerFilter:"input", width:160 },
    { title:"Categorización",  field:"pro_categorizacion", headerFilter:"input", width:150 },
  ];

  const fmt4 = cell => {
    const v = cell.getValue();
    if (v === null || v === '' || typeof v === 'undefined') return '';
    const n = Number(v);
    if (isNaN(n)) return '';
    return n.toFixed(4);
  };

  const dynTabCols = (dynCols||[]).map(c => {
    const isManual = manualNames.has(String(c.name)) || c.isManual === true;
    const col = { title: c.title, field: c.name, width: 180 };

    if (isManual) {
      col.editor = "input";      // editable
      col.validator = false;
    } else {
      col.hozAlign = "right";
      col.formatter = fmt4;      // formatos numéricos a 4 decimales
    }
    return col;
  });

  // ======== Inicializa Tabulator ========
  const table = new Tabulator("#matriz-grid", {
    layout: "fitDataStretch",
    height: "calc(100vh - 260px)",
    placeholder: "Cargando...",
    columns: [...baseCols, ...dynTabCols],

    // Paginación remota contra /data (estilo DataTables server-side)
    pagination: true,
    paginationMode: "remote",
    paginationSize: 100,
    progressiveLoad: "load", // scroll paginado (opcional)

    ajaxURL: urlData,
    ajaxConfig: { method: "POST", headers: {"X-Requested-With":"XMLHttpRequest"} },
    ajaxContentType: "form",
    ajaxParams: function () {
      // No dependas de params, lee desde la tabla (evita error de undefined)
      const page = (typeof this.getPage === 'function' && this.getPage()) || 1;
      const size = (typeof this.getPageSize === 'function' && this.getPageSize()) || 100;
      const start = (page - 1) * size;
      return {
        [csrfKey]: csrfVal,
        draw: 1,
        start: start,
        length: size,
        'search[value]': '' // puedes mapear tus filtros aquí
      };
    },
    ajaxResponse: function (url, _ignored, response) {
      // Backend devuelve {draw, recordsTotal, recordsFiltered, data}
      const total = (response && (response.recordsFiltered ?? response.recordsTotal)) || 0;
      const size  = (typeof this.getPageSize === 'function' && this.getPageSize()) || 100;
      const last  = Math.max(1, Math.ceil(total / size));
      return { data: (response && response.data) || [], last_page: last };
    },

    // Guardar inline + recálculo inmediato de la fila
    cellEdited: function (cell) {
      const row   = cell.getRow().getData();
      const name  = cell.getField();
      const value = cell.getValue();
      const proId = row.pro_id;

      if (!proId) return;
      if (!manualNames.has(name)) return; // solo guardamos manuales

      const payload = new URLSearchParams({ pro_id: proId, name, value });
      payload.append(csrfKey, csrfVal);

      fetch(urlSave, { method:"POST", body: payload })
        .then(r => r.json())
        .then(res => {
          // Recalcula en servidor (si tienes endpoint /eval)
          if (!urlEval) return null;

          const evalPayload = new URLSearchParams({
            pro_id: proId,
            overrides: JSON.stringify({[name]: value})
          });
          evalPayload.append(csrfKey, csrfVal);

          return fetch(urlEval, { method:"POST", body: evalPayload });
        })
        .then(r => r ? r.json() : null)
        .then(recalc => {
          if (recalc && recalc.row) {
            cell.getRow().update(recalc.row);
          }
        })
        .catch(err => console.error('save/eval error', err));
    },
  });

  // ======== Export CSV ========
  document.getElementById('btn-export').addEventListener('click', () => {
    table.download("csv", "matriz_<?= $matId ?>.csv");
  });
})();
</script>
